"""
Ejemplo sencillo de algoritmo genético en Python.

Este script busca maximizar la función f(x) = x * sin(10*x) en el intervalo [0, 1].
Se utiliza una representación real (número flotante) para cada individuo.
"""

import random
import math

def fitness(x: float) -> float:
    """Función de aptitud a maximizar."""
    return x * math.sin(10 * x)


def mutate(value: float, rate: float = 0.1) -> float:
    """
    Mutación: introduce una pequeña variación aleatoria en un individuo.

    Args:
        value: valor del individuo a mutar.
        rate: probabilidad de mutación.

    Returns:
        Individuo posiblemente mutado y limitado al intervalo [0, 1].
    """
    if random.random() < rate:
        value += random.uniform(-0.05, 0.05)
        # Aseguramos que el valor permanezca en el intervalo [0, 1]
        value = max(0.0, min(1.0, value))
    return value


def crossover(parent1: float, parent2: float) -> float:
    """Crossover de dos padres utilizando una combinación lineal aleatoria."""
    alpha = random.random()
    return alpha * parent1 + (1 - alpha) * parent2


def genetic_algorithm(
    population_size: int = 50,
    generations: int = 10,
    crossover_rate: float = 0.8,
    mutation_rate: float = 0.1,
) -> float:
    """
    Ejecuta un algoritmo genético sencillo para maximizar la función fitness.

    Args:
        population_size: número de individuos en la población.
        generations: número de generaciones a simular.
        crossover_rate: probabilidad de realizar crossover.
        mutation_rate: probabilidad de mutación.

    Returns:
        Mejor individuo encontrado.
    """
    # Inicializamos la población con valores aleatorios en [0, 1]
    population = [random.random() for _ in range(population_size)]

    for _ in range(generations):
        # Ordenamos la población por aptitud de mayor a menor
        population.sort(key=fitness, reverse=True)
        # Aplicamos elitismo: mantenemos los dos mejores individuos
        next_population = population[:2]
        # Generamos nuevos individuos hasta completar la población
        while len(next_population) < population_size:
            # Selección: elegimos padres entre los mejores 50 % de la población
            parent1 = random.choice(population[: population_size // 2])
            parent2 = random.choice(population[: population_size // 2])
            # Crossover con cierta probabilidad
            if random.random() < crossover_rate:
                child = crossover(parent1, parent2)
            else:
                child = parent1
            # Mutación
            child = mutate(child, mutation_rate)
            next_population.append(child)
        population = next_population

    # Devolvemos el mejor individuo encontrado
    return max(population, key=fitness)


if __name__ == "__main__":
    best = genetic_algorithm()
    print(f"Mejor solución: x = {best:.4f}")
    print(f"Valor de fitness: {fitness(best):.4f}")